<?php

namespace App\Http\Controllers;
use DB;
use Carbon\Carbon;
use App\Models\Sites;
use App\Models\LeaveRequest;
use App\Models\RosterAssign;
use Illuminate\Http\Request;
use App\General\GeoLoacation;
use App\Models\EmployeeBreak;
use App\Models\EmpTeamsMember;
use App\Models\RosterTemplate;
use App\Models\EmpCompanyDetails;
use App\Models\EmployeeAttendance;
use App\Models\EmpPersonalDetails;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use App\Models\AttendanceListChangeCol;
use App\Models\Fine;
use App\Models\Overtime;
use App\Models\Salary;
use App\Models\PublicHoliday;
use Illuminate\Support\Facades\Response;
use Illuminate\Support\Facades\Validator;


class AttendanceController extends Controller
{
    public function __construct()
    {
      $this->middleware('Permissions:Attendance Maintain')->only(['create', 'edit']);
    }

    public function index(Request $request)
    {
        $query = EmployeeAttendance::query();

        if($request->filled('filter'))
        {
          $query =  $this->filter(json_decode($request->filter, true),$query);
        }
  
        $query->with(['EmpPersonalDetails','EmpCompanyDetails','Sites','empTeamsMembers']);

        $count = $query->count();
        $start = $request->input('from', 0);
        $query->offset($start)->limit(10);
        $query->orderBy('id', 'desc');
        $query_result = $query->get();

        $data =  [
            'attendances' => $query_result,
        ];
        $array_filter =  json_decode($request->filter, true);
        $filters = [
          'first_name' => $array_filter['first_name'] ?? '', 
          'middle_name' => $array_filter['middle_name'] ?? '',
          'last_name' => $array_filter['first_name'] ?? '',
          'employee_email' => $array_filter['last_name'] ?? '',
          'compeleted' => $array_filter['first_name'] ?? '',
          'status' => $array_filter['compeleted'] ?? '',
        ];

        return response()->json([
            'message' => 'Get Attendance List Successfully',
            'data' => $data,
            'count' => $count,
            'filters' => $filters
        ],200);


    }

    public function filter($filters,$query)
    {
     
        foreach ($filters as $filterName => $filterValue) {
              if ($filterValue != null ||  $filterValue !="" ) {
                  switch ($filterName) {
                  case 'first_name':
                    $filterValue  = explode(" ",$filterValue);
                    
                    if(count($filterValue) == 1)
                    {
                        $query->whereHas('empPersonalDetails', function ($subquery) use ($filterValue) {
                            $subquery->where('first_name', 'like', '%' . $filterValue[0] . '%');
                            $subquery->orWhere('middle_name', 'like', '%' . $filterValue[0] . '%');
                            $subquery->orWhere('last_name', 'like', '%' . $filterValue[0] . '%');
                        });

                    }

                    if(count($filterValue) == 2)
                    {
                        $query->whereHas('empPersonalDetails', function ($subquery) use ($filterValue) {
                            $subquery->where('first_name', 'like', '%' . $filterValue[0] . '%');
                            $subquery->where('middle_name', 'like', '%' . $filterValue[1] . '%');
                            $subquery->where('last_name', 'like', '%' . $filterValue[1] . '%');
                        });

                    }

                    if(count($filterValue) == 3)
                    {

                        $query->whereHas('empPersonalDetails', function ($subquery) use ($filterValue) {
                            $subquery->where('first_name', 'like', '%' . $filterValue[0] . '%');
                            $subquery->where('middle_name', 'like', '%' . $filterValue[1] . '%');
                            $subquery->where('last_name', 'like', '%' . $filterValue[2] . '%');
                        });

                    }

                  break;
                  case 'teams':
                    $emp_teams_members = EmpTeamsMember::where('team_id', $filterValue)->pluck('emp_id')->toArray();
                    $query->whereIn('employee_id', $emp_teams_members);
                  break;
                  case 'user_type':
                    $query->whereHas('empCompanyDetails', function ($subquery) use ($filterValue) {
                    $filterValue == 0 ? 0 : 1;   
                    $subquery->where('user_type', 'like', '%' . $filterValue . '%');
                  });
                  break;
                  case 'title':
                    $query->whereHas('sites', function ($subquery) use ($filterValue) {
                    $subquery->where('title', 'like', '%' . $filterValue . '%');
                  });
                  break;
                  case 'Check_in':
                    $query->where('Check_in', 'like', '%' . $filterValue . '%');
                break; 
                  case 'Check_out':
                    $query->where('Check_out', 'like', '%' . $filterValue . '%');
                  break;  
                  case 'working_hours':
                    $query->where('working_hours', 'like', '%' . $filterValue . '%');
                  break; 
                  case 'date':
                   $filterValue = explode("/",$filterValue);
                   $query->whereBetween('date', [date("Y-m-d", strtotime(trim($filterValue[0]," "))), date("Y-m-d", strtotime(trim($filterValue[1]," ")))]);
                  break;    
                  case 'system_checkout':
                    if($filterValue != 2){
                        $query->where('checkout_type',$filterValue);
                    }                    
                   break;            
                  }
              }
        }

        return $query;
    } 

    public function create(Request $request)
    {
        $get_employes =  EmpCompanyDetails::with('EmpPersonalDetails')->where('del',0)->where('compeleted', 1)->orderBy('id', 'DESC')->get();
        $sites = Sites::where('del',0)->where('active',1)->get();
        return view('Attendance.create',compact('get_employes','sites'));
    }

    public function store(Request $request)
    {
        $employee = EmpCompanyDetails::find($request->employee);  

        if (!$employee) {
            return response()->json([
                'message' => 'Employee not found'
            ], 404);
        }

        if ($employee->status == '0') {
            return response()->json([
                'message' => 'The employee account is currently inactive'
            ], 400);
        }

        if ($employee->del == '1') {
            return response()->json([
                'message' => 'The employee account has been deleted'
            ], 400);
        }

        $validator = Validator::make($request->all(), [
            'employee' => 'required',
            'site' => 'required',
            'check_in' => 'required|date_format:H:i',
            'check_out' => 'nullable|date_format:H:i|after:check_in',
            'date' => 'required|date',
            'status' => 'nullable',
            'break_in' => ['nullable', 'array'],
            'break_out' => ['nullable', 'array'],
            'break_in.*' => ['nullable', 'date_format:H:i'],
            'break_out.*' => ['nullable', 'date_format:H:i', 'after_or_equal:break_in.*'],
        ], [
            'break_out.*.after_or_equal' => 'The break_out must be after or equal to break_in.',
            'check_out.after' => 'The check out time must be after check in time.',
        ]);

        if ($validator->fails()) {
            $error = $validator->errors()->first();

            return response()->json([
                'message' =>  $error
            ], 422);
        } else {
            $validatedData = $validator->validated();

            // Calculate total working hours
            $checkIn = Carbon::parse($validatedData['check_in']);
            $checkOut = isset($validatedData['check_out']) ? Carbon::parse($validatedData['check_out']) : null;

            $roster_assigns = DB::table('roster_assigns')
                ->join('roster_templates', 'roster_assigns.roster_template_id', '=', 'roster_templates.id')
                ->where('assign_to', $validatedData['employee'])
                ->where('schedule_date', $validatedData['date'])
                ->get();

            if (count($roster_assigns) == 0) {
                return response()->json([
                    'message' =>  'Roster not found',
                ], 400);
            }

            $roster = $roster_assigns[0];
            $startTime = Carbon::parse($roster->start_time);
            $endTime   = Carbon::parse($roster->end_time);

            if (!$checkIn->between($startTime, $endTime, true)) {
                return response()->json([
                    'message' => sprintf(
                        'Check‑in time must be between %s and %s.',
                        $startTime->format('H:i:s'),
                        $endTime->format('H:i:s')
                    )
                ], 422);
            }

            if ($checkOut && !$checkOut->between($startTime, $endTime, true)) {
                return response()->json([
                    'message' => sprintf(
                        'Check‑out time must be between %s and %s.',
                        $startTime->format('H:i:s'),
                        $endTime->format('H:i:s')
                    )
                ], 422);
            }

            $leave = LeaveRequest::where('employee_id', $request->employee)
                ->where('status', 1) // Approved leave only
                ->where('from', '<=', date('Y-m-d'))
                ->where('to', '>=', date('Y-m-d'))
                ->with(['leavepackage.leaveType'])
                ->first();

            $isWFH = false;
            $isShortLeave = false;

            if ($leave) {
                $leaveType = $leave->leavepackage ? $leave->leavepackage->leaveType : null;

                if ($leaveType) {
                    $leaveHours = (int) $leaveType->leave_hours;

                    if ($leaveHours === 0) {
                        $isWFH = true;
                    } elseif ($leaveHours > 0 && $leaveHours < 8) {
                        $isShortLeave = true;
                    } elseif ($leaveHours >= 8) {
                        return response()->json([
                            'message' => 'You are on leave. You are not allowed to check-in.'
                        ], 422);
                    }
                } else {
                    return response()->json([
                        'message' => 'You are on leave. You are not allowed to check-in.'
                    ], 422);
                }
            }

            if ($leave && !$isWFH && !$isShortLeave) {
                return response()->json([
                    'message' => 'You are on leave. You are not allowed to check-in.'
                ], 422);
            }
            $totalHours = $checkOut ? $checkOut->diffInMinutes($checkIn) : 0;
            // Calculate total breaks
            $totalBreaks = 0;
            if (isset($validatedData['break_in'])) {
                foreach ($validatedData['break_in'] as $key => $breakIn) {
                    if (isset($validatedData['break_out'][$key])) {
                        $breakInTime = Carbon::parse($breakIn);
                        $breakOutTime = Carbon::parse($validatedData['break_out'][$key]);
                        $totalBreaks += $breakOutTime->diffInMinutes($breakInTime);
                    }
                }
            }

            // Calculate total working hours with breaks
            $totalWorkingHours = $totalHours - $totalBreaks;

            // Lunch deduction logic: If check-out is after 14:00, subtract 60 minutes
            if ($checkOut && $checkOut->gt(Carbon::createFromTime(14, 0))) {
                $totalWorkingHours -= 60;
            }

            // Ensure totalWorkingHours doesn't go negative
            $totalWorkingHours = max($totalWorkingHours, 0);

            $insertedRecord = EmployeeAttendance::create([
                'employee_id' => $validatedData['employee'],
                'roster_template_id' => $roster->id,
                'site_id' => $validatedData['site'],
                'check_in' => $validatedData['check_in'],
                'working_hours' => $checkOut ? $totalWorkingHours : '',
                'check_out' => $validatedData['check_out'],
                'date' => $validatedData['date'],
                'status' => $validatedData['status'] ?? null,
                'added_by' => Auth::user()->id
            ]);

            $id = $insertedRecord->id;
            if (isset($validatedData['break_in'])) {
                foreach ($validatedData['break_in'] as $key => $breakIn) {
                    if ($breakIn !== null && isset($validatedData['break_out'][$key])) {
                        $breakOut = $validatedData['break_out'][$key];
                        EmployeeBreak::create([
                            'emp_attendance_id' => $id,
                            'break_in' => $breakIn,
                            'break_out' => $breakOut,
                        ]);
                    }
                }
            }
            $this->applyLateFineOnCheckIn($insertedRecord, $roster_assigns);

            session()->flash('success', 'Attendance Saved Successfully');
            return response()->json([
                'message' => 'Attendance Saved Successfully',
                'data'   => $insertedRecord
            ], 200);
        }
    }

    public function edit($id)
    {
        $get_employes =  EmpCompanyDetails::with('EmpPersonalDetails')->where('del',0)->where('compeleted', 1)->orderBy('id', 'DESC')->get();
        $sites = Sites::where('del',0)->where('active',1)->get();
        $emp_attend = EmployeeAttendance::where('id',$id)->first();
        $emp_breaks = EmployeeBreak::where('emp_attendance_id',$id)->get();
        return view('Attendance.edit', compact('get_employes', 'sites','emp_attend','emp_breaks'));
    }
 
    public function update(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'employee' => 'required',
            'site' => 'required',
            'check_in' => 'required|date_format:H:i',
            'check_out' => 'nullable|date_format:H:i|after:check_in',
            'date' => 'required|date',
            'status' => 'nullable',
            'break_in' => ['nullable', 'array'],
            'break_out' => ['nullable', 'array'],
            'break_in.*' => ['nullable', 'date_format:H:i'],
            'break_out.*' => ['nullable', 'date_format:H:i', 'after_or_equal:break_in.*'],
        ], [
            'break_out.*.after_or_equal' => 'The break_out must be after or equal to break_in.',
            'check_out.after' => 'The check out time must be after check in time.',
        ]);

        if ($validator->fails()) {
            $error = $validator->errors()->first();

            return response()->json([
                'message' =>  $error
            ], 422);
        } else {
            $employeeAttendance = EmployeeAttendance::findOrFail($id);
            $validatedData = $validator->validated();

            $roster_assigns = DB::table('roster_assigns')
                ->where('assign_to', $validatedData['employee'])
                ->where('schedule_date', $validatedData['date'])
                ->count();

            if ($roster_assigns == 0) {
                return response()->json([
                    'message' =>  'Roster not found',
                ], 400);
            }

            // Parse check-in and check-out times
            $checkIn = Carbon::parse($validatedData['check_in']);
            $checkOut = isset($validatedData['check_out']) ? Carbon::parse($validatedData['check_out']) : null;

            // Calculate total hours in minutes
            $totalHours = $checkOut ? $checkOut->diffInMinutes($checkIn) : 0;

            // Calculate total breaks
            $totalBreaks = 0;
            if (isset($validatedData['break_in'])) {
                foreach ($validatedData['break_in'] as $key => $breakIn) {
                    if (isset($validatedData['break_out'][$key])) {
                        $breakInTime = Carbon::parse($breakIn);
                        $breakOutTime = Carbon::parse($validatedData['break_out'][$key]);
                        $totalBreaks += $breakOutTime->diffInMinutes($breakInTime);
                    }
                }
            }

            // Calculate total working hours with breaks
            $totalWorkingHours = $totalHours - $totalBreaks;

            // **New Logic: Deduct 60 minutes if check_out is after 14:00**
            if ($checkOut && $checkOut->gt(Carbon::createFromTime(14, 0))) {
                $totalWorkingHours -= 60;
            }

            // Ensure totalWorkingHours doesn't go negative
            $totalWorkingHours = max($totalWorkingHours, 0);

            $employeeAttendance->update([
                'employee_id' => $validatedData['employee'],
                'site_id' => $validatedData['site'],
                'working_hours' => $validatedData['check_out'] ? $totalWorkingHours : '',
                'check_in' => $validatedData['check_in'],
                'check_out' => $validatedData['check_out'],
                'status' => $validatedData['status'],
                'date' => $validatedData['date'],
                'added_by' => Auth::user()->id
            ]);

            // Delete existing breaks
            $employeeAttendance->breaks()->delete();

            if (isset($validatedData['break_in'])) {
                foreach ($validatedData['break_in'] as $key => $breakIn) {
                    $breakOut = isset($validatedData['break_out'][$key]) ? $validatedData['break_out'][$key] : null;

                    EmployeeBreak::create([
                        'emp_attendance_id' => $employeeAttendance->id,
                        'break_in' => $breakIn,
                        'break_out' => $breakOut,
                    ]);
                }
            }

            session()->flash('success', 'Attendance Updated Successfully');
            return response()->json([
                'message' => 'Attendance Updated Successfully',
                'data' => $employeeAttendance
            ], 200);
        }
    }

    public function destroy($id)
    {
        EmployeeAttendance::where('id',$id)->delete();
        EmployeeBreak::where('emp_attendance_id',$id)->delete();
        session()->flash('success', 'Attendance Deleted Successfully');
        return response()->json([
            'message' => 'Attendance Deleted Successfully'
        ],200);
    }

    public function deletebreak($id)
    {
        $empbreak = EmployeeBreak::where('id', $id)->first();
    
        $breakIn = Carbon::parse($empbreak->break_in);
        $breakOut = Carbon::parse($empbreak->break_out);
        $totalbreakHours = $breakOut->diffInHours($breakIn);
    
        $attendance = EmployeeAttendance::where('id', $empbreak->emp_attendance_id)->first();
        $totalhours = $attendance->working_hours + $totalbreakHours;
    
        $attendance->update([
            'working_hours' => $totalhours,
        ]);
    
        EmployeeBreak::where('id', $id)->delete();
        session()->flash('success_message', 'Break Deleted Successfully');
        return response()->json([
            'message' => 'Break Deleted Successfully'
        ], 200);
    }
    
    public function attendance_request(Request $request)
    {
        $query = EmployeeAttendance::query();

        if($request->filled('filter'))
        {
          $query =  $this->filter_request(json_decode($request->filter, true),$query);
        }
  
        $query->with(['EmpPersonalDetails','EmpCompanyDetails','Sites','addedby']);

        $count = $query->count();
        $start = $request->input('from', 0);
        $query->offset($start)->limit(10);
        $query->orderBy('id', 'desc');
        $query_result = $query->get();

        $data =  [
            'attendances' => $query_result,
        ];
        $array_filter =  json_decode($request->filter, true);
        $filters = [
          'first_name' => $array_filter['first_name'] ?? '', 
          'middle_name' => $array_filter['middle_name'] ?? '',
          'last_name' => $array_filter['first_name'] ?? '',
          'employee_email' => $array_filter['last_name'] ?? '',
          'compeleted' => $array_filter['first_name'] ?? '',
          'status' => $array_filter['compeleted'] ?? '',
        ];

        return response()->json([
            'message' => 'Get Attendance List Successfully',
            'data' => $data,
            'count' => $count,
            'filters' => $filters
        ],200);

    }

    public function filter_request($filters,$query)
    { 
        foreach ($filters as $filterName => $filterValue) {
              if ($filterValue != null ||  $filterValue !="" ) {
                  switch ($filterName) {
                    case 'first_name':
                        $filterValue  = explode(" ",$filterValue);

                        if(count($filterValue) == 1)
                        {
                            $query->whereHas('empPersonalDetails', function ($subquery) use ($filterValue) {
                                $subquery->where('first_name', 'like', '%' . $filterValue[0] . '%');
                                $subquery->orWhere('middle_name', 'like', '%' . $filterValue[0] . '%');
                                $subquery->orWhere('last_name', 'like', '%' . $filterValue[0] . '%');
                            });

                        }

                        if(count($filterValue) == 2)
                        {
                            $query->whereHas('empPersonalDetails', function ($subquery) use ($filterValue) {
                                $subquery->where('first_name', 'like', '%' . $filterValue[0] . '%');
                                $subquery->where('middle_name', 'like', '%' . $filterValue[1] . '%');
                                $subquery->where('last_name', 'like', '%' . $filterValue[1] . '%');
                            });

                        }

                        if(count($filterValue) == 3)
                        {

                            $query->whereHas('empPersonalDetails', function ($subquery) use ($filterValue) {
                                $subquery->where('first_name', 'like', '%' . $filterValue[0] . '%');
                                $subquery->where('middle_name', 'like', '%' . $filterValue[1] . '%');
                                $subquery->where('last_name', 'like', '%' . $filterValue[2] . '%');
                            });

                        }
    
                    break;
                  case 'user_type':
                    $query->whereHas('empCompanyDetails', function ($subquery) use ($filterValue) {
                    $filterValue == 0 ? 0 : 1;   
                    $subquery->where('user_type', 'like', '%' . $filterValue . '%');
                  });
                  break;
                  case 'first_name_add':

                    $filterValue  = explode(" ",$filterValue);
                        if(count($filterValue) == 1)
                        {
                            $query->whereHas('addedby', function ($subquery) use ($filterValue) {
                                $subquery->where('first_name', 'like', '%' . $filterValue[0] . '%');
                            });
                        }
    
                        if(count($filterValue) == 2)
                        {
                            $query->whereHas('addedby', function ($subquery) use ($filterValue) {
                                $subquery->where('first_name', 'like', '%' . $filterValue[0] . '%');
                                $subquery->where('last_name', 'like', '%' . $filterValue[1] . '%');
                            });
                        }
    
                        if(count($filterValue) == 3)
                        {
                            $query->whereHas('addedby', function ($subquery) use ($filterValue) {
                                $subquery->where('first_name', 'like', '%' . $filterValue[0] . '%');
                                $subquery->where('middle_name', 'like', '%' . $filterValue[1] . '%');
                                $subquery->where('last_name', 'like', '%' . $filterValue[2] . '%');
                            });
                        }
                   
                  break;
                  case 'title':
                    $query->whereHas('sites', function ($subquery) use ($filterValue) {
                    $subquery->where('title', 'like', '%' . $filterValue . '%');
                  });
                  break;
                  case 'Check_in':
                    $query->where('Check_in', 'like', '%' . $filterValue . '%');
                break; 
                  case 'Check_out':
                    $query->where('Check_out', 'like', '%' . $filterValue . '%');
                  break;  
                  case 'working_hours':
                    $query->where('working_hours', 'like', '%' . $filterValue . '%');
                  break; 
                  case 'status':
                      $filterValue =   $filterValue ? 1 : 0;
                      $query->where('status', 'like', '%' . $filterValue . '%');
                  break; 
                  case 'date':
                    $filterValue = explode("/",$filterValue);
                    $query->whereBetween('date', [date("Y-m-d", strtotime(trim($filterValue[0]," "))), date("Y-m-d", strtotime(trim($filterValue[1]," ")))]);
                   break;                 
                  }
              }
        }

        return $query;
    } 

    public function updateStatus($id)
    {     
        $EmployeeAttendance = EmployeeAttendance::find($id);

        if (!$EmployeeAttendance) {
            return response()->json([
                'message' => 'Record not found',
            ], 404);
        }
        // Toggle the active status
        $EmployeeAttendance->status = $EmployeeAttendance->status == 1 ;
        $EmployeeAttendance->save();
        session()->flash('success', 'Status Updated Successfully');
        return response()->json([
            'message' => 'Status Updated Successfully',
            'data' => [
                'active' => $project->active,
            ],
        ], 200);
    }

    public function approved_all(Request $request)
    {
        EmployeeAttendance::query()->update(['status' => 1]);
        session()->flash('success', 'All Request Are Approved Successfully');
    
        return response()->json([
            'message' => 'All Request Are Approved Successfully',
            'data' => [
                'status' => 1,
            ],
        ], 200);
    }
    
    public function approveSelected(Request $request)
    {
        $selectedIds = $request->input('ids', []);

        EmployeeAttendance::whereIn('id', $selectedIds)->update(['status' => 1]);

        session()->flash('success', 'Approved Selected Request Successfully');
    
        return response()->json([
            'message' => 'Approved Selected Request Successfully',
            'data' => [
                'status' => 1,
            ],
        ], 200);
    }

    public function attendance_portal(Request $request)
    {
        $sites = Sites::where('del',0)->where('active',1)->get();
        return view('Attendance.attendanceRecord',compact('sites'));
    }
    
    public function attendance_portal_employee(Request $request)
    {
        $validator = Validator::make($request->all(), [
                'from' => 'required',
                'to' => 'required|after_or_equal:from',
                'site_id' => 'nullable',
                'employee_name' => 'nullable'
            ],
        [
            "after_or_equal" => "To date must be bigger or equal to From"
        ]);
    
        if ($validator->fails()) {
            $error = $validator->errors()->first();
    
            return response()->json([
                'message' =>  $error
            ], 422);
            
        } else {
            $validatedData = $validator->validated();
            $required_minutes = 0;
            
            $required_minutes = DB::table('roster_assigns')->whereBetween('roster_assigns.schedule_date',[$validatedData['from'], $validatedData['to']])
            ->join('roster_templates','roster_assigns.roster_template_id','roster_templates.id')
            ->sum('roster_templates.working_hours');

            

            $employee_attendances = EmployeeAttendance::whereBetween('date', [$validatedData['from'], $validatedData['to']]);

            if ($validatedData['site_id']) {
                $employee_attendances->where('site_id', $validatedData['site_id']);
            }

            if ($validatedData['employee_name']) {
                $filterValue = explode(" ", $validatedData['employee_name']);
                
                $employee_attendances->whereHas('empPersonalDetails', function ($subquery) use ($filterValue) {
                    // Handle one name input
                    if (count($filterValue) == 1) {
                        $subquery->where(function ($query) use ($filterValue) {
                            $query->where('first_name', 'like', '%' . $filterValue[0] . '%')
                                ->orWhere('middle_name', 'like', '%' . $filterValue[0] . '%')
                                ->orWhere('last_name', 'like', '%' . $filterValue[0] . '%');
                        });
                    }

                    // Handle two names input
                    if (count($filterValue) == 2) {
                        $subquery->where(function ($query) use ($filterValue) {
                            $query->where('first_name', 'like', '%' . $filterValue[0] . '%')
                                ->where(function ($query) use ($filterValue) {
                                    $query->where('middle_name', 'like', '%' . $filterValue[1] . '%')
                                            ->orWhere('last_name', 'like', '%' . $filterValue[1] . '%');
                                });
                        });
                    }

                    // Handle three names input
                    if (count($filterValue) == 3) {
                        $subquery->where('first_name', 'like', '%' . $filterValue[0] . '%')
                                ->where('middle_name', 'like', '%' . $filterValue[1] . '%')
                                ->where('last_name', 'like', '%' . $filterValue[2] . '%');
                    }
                });
            }

            // Clone the query for working minutes
            $working_minutes = (clone $employee_attendances)->sum('working_hours');

            // Clone the query for distinct employee count
            $employee_count = (clone $employee_attendances)
                ->select(DB::raw('COUNT(DISTINCT employee_id) as count'))
                ->first(); // Use first() to get a single row containing the count

            // Clone the query for employee attendance lists
            $employee_attendances_lists = (clone $employee_attendances)
                ->distinct()
                ->pluck('employee_id');
            
            
            $employee_attendances = DB::table('employee_attendances')
                ->join('emp_personal_details', 'employee_attendances.employee_id', '=', 'emp_personal_details.emp_id')
                ->join('emp_company_details', 'employee_attendances.employee_id', '=', 'emp_company_details.id')
                ->join('roles', 'emp_company_details.access_role', '=', 'roles.code')
                ->whereIn('employee_attendances.employee_id', $employee_attendances_lists)
                ->whereBetween('date', [$validatedData['from'], $validatedData['to']]);
            
            if ($validatedData['site_id']) {
                $employee_attendances->where('employee_attendances.site_id', $validatedData['site_id']);
            }

            if ($validatedData['employee_name']) {
                $filterValue = explode(" ", $validatedData['employee_name']);

                $employee_attendances->where(function ($query) use ($filterValue) {
                    // Handle one name input
                    if (count($filterValue) == 1) {
                        $query->where('emp_personal_details.first_name', 'like', '%' . $filterValue[0] . '%')
                            ->orWhere('emp_personal_details.middle_name', 'like', '%' . $filterValue[0] . '%')
                            ->orWhere('emp_personal_details.last_name', 'like', '%' . $filterValue[0] . '%');
                    }

                    // Handle two names input
                    if (count($filterValue) == 2) {
                        $query->where('emp_personal_details.first_name', 'like', '%' . $filterValue[0] . '%')
                            ->where(function ($subquery) use ($filterValue) {
                                $subquery->where('emp_personal_details.middle_name', 'like', '%' . $filterValue[1] . '%')
                                        ->orWhere('emp_personal_details.last_name', 'like', '%' . $filterValue[1] . '%');
                            });
                    }

                    // Handle three names input
                    if (count($filterValue) == 3) {
                        $query->where('emp_personal_details.first_name', 'like', '%' . $filterValue[0] . '%')
                            ->where('emp_personal_details.middle_name', 'like', '%' . $filterValue[1] . '%')
                            ->where('emp_personal_details.last_name', 'like', '%' . $filterValue[2] . '%');
                    }
                });
            }
            
            // Group by employee_id and aggregate the other columns
            $employee_attendances
                ->groupBy('employee_attendances.employee_id') // Group by employee_id
                ->select(
                    'employee_attendances.employee_id as emp_id',
                    DB::raw('MAX(emp_personal_details.first_name) as first_name'), // Aggregate first_name
                    DB::raw('MAX(emp_personal_details.middle_name) as middle_name'), // Aggregate middle_name
                    DB::raw('MAX(emp_personal_details.last_name) as last_name'), // Aggregate last_name
                    DB::raw('MAX(emp_personal_details.image) as image'), // Aggregate image
                    DB::raw('MAX(emp_company_details.user_type) as user_type'), // Aggregate user_type
                    DB::raw('MAX(roles.title) as title') // Aggregate role_title
                );
            
                // ->select('employee_attendances.employee_id',
                // 'employee_attendances.site_id',
                // 'emp_personal_details.*',
                // 'emp_company_details.user_type as user_type',
                // 'roles.title')
            

            $count = $employee_count->count;
            $start = $request->input('page', 0);
            $employee_attendances->offset($start)->limit(10);
            $query_result = $employee_attendances->get();

	    	return response()->json([
                'message' => 'Get Attendance List Successfully',
                'data' => $query_result,
                'workinghours' => convertMinutesToHours($working_minutes),
                'requirehours' => convertMinutesToHours($required_minutes),
                'employe'=> $employee_count->count,
            ], 200);
        }
    }
    
    public function employe_attendance(Request $request)
    {
        $request->validate([
            'to'     => 'required|date',
            'from'   => 'required|date|before_or_equal:to',
            'empid'  => 'required',
            'site_id'=> 'nullable'
        ]);

        $toDate   = $request->input('to');
        $fromDate = $request->input('from');
        $siteid   = $request->input('site_id');
        $empid    = $request->input('empid');

        if ($siteid) {
            $attendance = EmployeeAttendance::where('site_id', $siteid)
                ->where('employee_id', $empid)
                ->whereBetween('date', [$fromDate, $toDate])
                ->with('RosterAssign.RosterTemplate', 'sites','rosterTemplate')
                ->with('breaks')
                ->get();

            $overtime = Overtime::where('site_id', $siteid)
                ->where('employee_id', $empid)
                ->whereBetween('date', [$fromDate, $toDate])
                ->with('sites')
                ->where('status', 2)
                ->get();
        } else {
            $attendance = EmployeeAttendance::where('employee_id', $empid)
                ->whereBetween('date', [$fromDate, $toDate])
                ->with('RosterAssign.RosterTemplate', 'sites','rosterTemplate')
                ->with('breaks')
                ->get();

            $overtime = Overtime::where('employee_id', $empid)
                ->whereBetween('date', [$fromDate, $toDate])
                ->with('sites')
                ->where('status', 2)
                ->get();
        }

        $attendance->transform(function ($item) {
            $item->flag = 'attendance';  
            return $item;
        });

        $overtime->transform(function ($item) {
            $item->flag = 'overtime';  
            return $item;
        });

        $merged = $attendance->merge($overtime);


        $merged = $merged->sortByDesc('date')->values();
        return response()->json([
            'message' => 'Get Employee Attendance and Overtime List Successfully',
            'data'    => $merged,
        ], 200);
    }
    ////////////////Attendance Api///////////////
    public function single_employee_attendance(Request $request)
    {
        // Validate the incoming request inputs.
        $request->validate([
            'to'      => 'required|date',
            'from'    => 'required|date|before_or_equal:to',
            'site_id' => 'nullable'
        ]);

        // Retrieve input values.
        $toDate   = $request->input('to');
        $fromDate = $request->input('from');
        $siteid   = $request->input('site_id');

        // Get the authenticated user's employee id.
        $empid = $request->user()->id;

        // Retrieve attendance and overtime data conditionally based on site_id.
        if ($siteid) {
            $attendance = EmployeeAttendance::where('site_id', $siteid)
                ->where('employee_id', $empid)
                ->whereBetween('date', [$fromDate, $toDate])
                ->with('RosterAssign.RosterTemplate', 'sites')
                ->with('breaks')
                ->orderBy('date', 'desc')
                ->get();

            $overtime = Overtime::where('site_id', $siteid)
                ->where('employee_id', $empid)
                ->whereBetween('date', [$fromDate, $toDate])
                ->with('sites')
                ->where('status', 2) 
                ->orderBy('date', 'desc')
                ->get();
        } else {
            $attendance = EmployeeAttendance::where('employee_id', $empid)
                ->whereBetween('date', [$fromDate, $toDate])
                ->with('RosterAssign.RosterTemplate', 'sites')
                ->with('breaks')
                ->orderBy('date', 'desc')
                ->get();

            $overtime = Overtime::where('employee_id', $empid)
                ->whereBetween('date', [$fromDate, $toDate])
                ->with('sites')
                ->where('status', 2) 
                ->orderBy('date', 'desc')
                ->get();
        }
    
        // Add flag to each attendance record.
        $attendance->transform(function ($item) {
            $item->flag = 'attendance';
            return $item;
        });

        $overtime = $overtime->map(function ($item) {
            $item->flag = 'overtime';
            
            // Format dates to show only time when assigning to the response object
            $item->check_in = Carbon::parse($item->check_in)->timezone('Asia/Karachi')->format('H:i:s');
            $item->check_out = Carbon::parse($item->check_out)->timezone('Asia/Karachi')->format('H:i:s');
            
            return $item;
        });
    
    
        // Merge both collections and sort them by date descending.
        $merged = $attendance->merge($overtime)->sortByDesc('date')->values();

        $responseData = $merged->map(function ($item) {
            return [
                'id' => $item->id,
                'employee_id' => $item->employee_id,
                'site_id' => $item->site_id,
                'check_in' => $item->check_in,  // Already formatted to 'H:i:s'
                'check_out' => $item->check_out,
                'date' => $item->date,
                'working_hours' => $item->working_hours,
                'status' => $item->status,
                'description' => $item->description,
                'created_at' => $item->created_at->format('Y-m-d H:i:s'),
                'updated_at' => $item->updated_at->format('Y-m-d H:i:s'),
                'flag' => $item->flag,
                'sites' => $item->sites
            ];
        });
        
        // Return the manually constructed response
        return response()->json([
            'message' => 'Get Employee Attendance and Overtime List Successfully',
            'data'    => $responseData,
        ], 200);
    }


    public function check_in(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'employee_id' => 'required',
            'longitude' => 'required',
            'latitude' => 'required',
            'date' => 'required|date',
            'check_in' => 'date_format:H:i',
            'check_out' => 'date_format:H:i|after:check_in',
            'site_id' => 'nullable',
        ]);

        if ($validator->fails()) {
            $error = $validator->errors()->first();

            return response()->json([
                'message' => $error
            ], 422);
        } else {
            $validatedData = $validator->validated();

            $check_leave = LeaveRequest::where('employee_id',$request->employee_id)->whereIn('status',[1,4])->where('from',">=",date('Y-m-d'))
            ->where('to',"<=",date('Y-m-d'))->exists();
            if($check_leave){
            return response()->json([
            'message' => 'Your are on leave. You are not allowed to check-in'
            ], 422);
            }

            $site = Sites::whereRaw("CONVERT(longitude, CHAR) LIKE CONCAT('%', ?, '%')", [$validatedData['longitude']])
            ->whereRaw("CONVERT(latitude, CHAR) LIKE CONCAT('%', ?, '%')", [$validatedData['latitude']])
            ->select('id')
            ->first();

            $site_id = $site ? $site->id : null;

            $data = [
                'employee_id' => $validatedData['employee_id'],
                'longitude' => $validatedData['longitude'],
                'latitude' =>  $validatedData['latitude'],
                'date' => $validatedData['date'],
                'site_id' => $site_id,
            ];

            if ($request->has('check_in')) {
                $data['check_in'] = $request->input('check_in');
            }

            if ($request->has('check_out')) {
                $data['check_out'] = $request->input('check_out');
            }

            $existingData = EmployeeAttendance::where('employee_id', $data['employee_id'])
                ->where('date', $data['date'])
                ->first();

            if ($request->has('check_out')) {
                if ($existingData && !$existingData->check_in) {
                    return response()->json([
                        'message' => 'Cannot check out without checking in first.'
                    ], 422);
                }

                if ($existingData) {
                    $checkIn = Carbon::parse($existingData->check_in);
                    $checkOut = Carbon::parse($validatedData['check_out']);
                    $totalHours = $checkOut->diffInMinutes($checkIn);
                    $existingData->update([
                        'check_out' => $data['check_out'],
                        'working_hours' => $totalHours
                    ]);
                    $message = 'Check Out successfully';
                } else {
                    return response()->json([
                        'message' => 'Cannot check out without checking in first.'
                    ], 422);
                }
            } else {
                if ($existingData) {
                    $message = 'Check In already exists';
                } else {
                    EmployeeAttendance::create($data);
                    $message = 'Check In successfully';
                }
            }

            return response()->json([
                'message' => $message,
                'data' => $existingData
            ], 200);
        }
    }

    public function _check_in(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'employee_id' => 'required',
            'longitude' => 'required',
            'latitude' => 'required',
            'check_in' => 'date_format:H:i',
            'site_id' => 'nullable',
            'image' => 'required|mimes:jpeg,png,jpg,gif|max:7168',
        ]);
    
        if ($validator->fails()) {
            $error = $validator->errors()->first();
    
            return response()->json([
                'message' => $error
            ], 422);
        } else {
            $validatedData = $validator->validated();
            if(Auth::user()->id != $validatedData['employee_id']){
                return response()->json([
                    'message' => "Invalid Request."
                ], 422);
            }

            // --- LEAVE CHECK FOR WFH LOGIC --
            $today = now()->format('Y-m-d');
                $leave = LeaveRequest::where('employee_id', Auth::user()->id)
                ->where('status', 1)
                ->where('from', '<=', $today)
                ->where('to', '>=', $today)
                ->with(['leavepackage.leaveType'])
                ->first();
            $isWFH = false;
            $isShortLeave = false;
            if ($leave) {
                $leaveType = $leave->leavepackage ? $leave->leavepackage->leaveType : null;
                if ($leaveType) {
                    $leaveHours = (int) $leaveType->leave_hours;
                    if ($leaveHours === 0) {
                        $isWFH = true; // Work From Home
                    } elseif ($leaveHours > 0 && $leaveHours < 8) {
                        $isShortLeave = true; // Short Leave
                    } else {
                        // Full day leave, block check-in
                        return response()->json([
                            'message' => 'Your are on leave. You are not allowed to check-in'
                        ], 422);
                    }
                } else {
                    // No leave type found, block check-in
                    return response()->json([
                        'message' => 'Your are on leave. You are not allowed to check-in'
                    ], 422);
                }
            }

            $currentTime = now()->format('H:i:s');
            // $currentTime = '16:40';
            $check_roster = RosterAssign::join('roster_templates','roster_assigns.roster_template_id','roster_templates.id')
                                ->where('roster_assigns.assign_to',Auth::user()->id)
                                ->where('schedule_date',date('Y-m-d'))
                                ->where('roster_templates.start_time',"<=", $currentTime)
                                ->where('roster_templates.end_time',">=", $currentTime)
                                ->first();
            if(!$check_roster && !$isWFH){
                $message = "Please clock in according to your scheduled roster time.";

                /////////////////// ATTENDANCE LOGS ////////////////////
                $log = [
                    'employee_id' => Auth::user()->id,
                    'site_id' => null,
                    'check_in' => $currentTime,
                    'check_out' => null,
                    'date' => formatDate($today),
                    'longitude' => $validatedData['longitude'],
                    'latitude' => $validatedData['latitude'],
                    'message' => $message,
                    'error_type' => 'invalid-roster-time',
                    'request_type' => 'attendance'
                ];
                storeAttendanceLogs($log);
                /////////////////// ATTENDANCE LOGS END ////////////////////
                return response()->json([
                    'message' => $message
                ], 422);
            }

            $foundSite = null;
            if (!$isWFH) {
                $point = [
                    'longitude' => $validatedData['longitude'],
                    'latitude' => $validatedData['latitude'],
                ];
                $sites = Sites::where('del',0)->where('active',1)->get();
                foreach ($sites as $site) {
                    $center = [
                        'longitude' => $site->longitude,
                        'latitude' => $site->latitude,
                    ];
                    $radius = $site->area_radius; 
                    if (GeoLoacation::isPointWithinRadius($point, $center, $radius)) {
                        $foundSite = $site;
                        break;
                    }
                }
                if (!$foundSite) {
                    $data = [
                        'employee_location' => $point,
                        'all_sites' => $sites
                    ];
                    $message = 'The provided location is outside the permitted radius of any site.';
                    /////////////////// ATTENDANCE LOGS ////////////////////
                    $log = [
                        'employee_id' => Auth::user()->id,
                        'site_id' => null,
                        'check_in' => $currentTime,
                        'check_out' => null,
                        'date' => formatDate($today),
                        'longitude' => $validatedData['longitude'],
                        'latitude' => $validatedData['latitude'],
                        'message' => $message,
                        'error_type' => 'invalid-location',
                        'request_type' => 'attendance'
                    ];
                    storeAttendanceLogs($log);
                    /////////////////// ATTENDANCE LOGS END ////////////////////
                    return response()->json([
                        'message' => $message
                    ], 422);
                }
            }

            $previousRecord = EmployeeAttendance::where('employee_id', Auth::user()->id)
                ->where('date', $today)
                ->latest()
                ->first();
    
            if ($previousRecord && !$previousRecord->check_out) {
                $message = 'You are already checked in at a site. Please check out first before checking in again.';

                /////////////////// ATTENDANCE LOGS ////////////////////
                $log = [
                    'employee_id' => Auth::user()->id,
                    'site_id' => $isWFH ? null : $foundSite->id,
                    'check_in' => $currentTime,
                    'check_out' => null,
                    'date' => formatDate($today),
                    'longitude' => $validatedData['longitude'],
                    'latitude' => $validatedData['latitude'],
                    'message' => $message,
                    'error_type' => 'already-check-in',
                    'request_type' => 'attendance'
                ];
                storeAttendanceLogs($log);
                /////////////////// ATTENDANCE LOGS END ////////////////////
                return response()->json([
                    'message' => $message,
                    'checkin_id' => $previousRecord->id
                ], 422);
            }



            $validatedData['image'] = "";
            if ($request->hasFile('image')) 
            {

                $image = $request->file('image');
                $imageName = Auth::user()->id.time() . '.' . $image->getClientOriginalExtension();
                $image->move(public_path('upload/images'), $imageName);
                $validatedData['image'] = 'upload/images/' . $imageName;

            }
            // dd($isWFH);
            $data = [
                'employee_id'=> Auth::user()->id,
                'date'       => $today,
                'check_in'   => $currentTime,
                'longitude'  => $validatedData['longitude'],
                'latitude'   => $validatedData['latitude'],
                'status'     => 1,
                'site_id'    => $isWFH ? null : $foundSite->id,
                'added_by'   => Auth::user()->id,
                'image'      => $validatedData['image'],
                'roster_template_id' => $check_roster ? $check_roster->roster_template_id : null,
            ];
            if ($isWFH) {
                $data['description'] = 'Work From Home';
            }
            $checkin = EmployeeAttendance::create($data);
            if (!$isWFH) {
                $this->applyLateFineOnCheckIn($checkin, $check_roster);
            }
            $message = 'Check-in Successfully';
            if ($isShortLeave) {
                $message .= ' (You are on short leave)';
            }
            ///////////////// ATTENDANCE LOGS ////////////////////
            $log = [
                'employee_id' => Auth::user()->id,
                'site_id' => $isWFH ? null : $foundSite->id,
                'check_in' => $currentTime,
                'check_out' => null,
                'date' => formatDate($today),
                'longitude' => $validatedData['longitude'],
                'latitude' => $validatedData['latitude'],
                'message' => $message,
                'error_type' => 'success',
                'request_type' => 'attendance'
            ];
            storeAttendanceLogs($log);
            /////////////////// ATTENDANCE LOGS END ////////////////////
            return response()->json([
                'message' => $message,
                'data' => $checkin
            ], 200);
        }
    }
    public function applyLateFineOnCheckIn($attendance, $roster)
    {
        try {
            // Check if employee has any leave request (pending or approved) - if so, don't apply late fine
            $today = $attendance->date;
            $leave = LeaveRequest::where('employee_id', $attendance->employee_id)
                ->whereIn('status', [0, 1, 4]) // 0 = pending, 1 = approved, 4 = approved as unpaid
                ->where('from', '<=', $today)
                ->where('to', '>=', $today)
                ->with(['leavepackage.leaveType'])
                ->first();
            
            if ($leave) {
                $leaveType = $leave->leavepackage ? $leave->leavepackage->leaveType : null;
                if ($leaveType) {
                    $leaveHours = (int) $leaveType->leave_hours;
                    if ($leaveHours > 0 && $leaveHours < 8) {
                        // Employee has pending or approved short leave, don't apply late fine
                        return;
                    }
                }
            }

            // A fine should only be applied to the very first check-in of the day, if it's late.
            $firstCheckInForDay = EmployeeAttendance::where('employee_id', $attendance->employee_id)
                ->where('date', $attendance->date)
                ->orderBy('check_in', 'asc')
                ->first();

            // If the current attendance record is not the first check-in of the day, do not apply a fine.
            if ($firstCheckInForDay->id !== $attendance->id) {
                return;
            }
            
            $salary = Salary::where('employee_id', $attendance->employee_id)
                ->where(function ($query) use ($attendance) {
                    $query->where('from', '<=', \Carbon\Carbon::parse($attendance->date)->format('Y-m'))
                        ->where('to', '>=', \Carbon\Carbon::parse($attendance->date)->format('Y-m'));
                })
                ->first();

            if (!$salary) {
                return; // No salary found, skip fine calculation
            }

            $basicSalary = $salary->basic_salary;
            
            $startOfMonth = Carbon::parse($attendance->date)->startOfMonth();
            $endOfMonth = Carbon::parse($attendance->date)->endOfMonth();
            
            // Get holidays for the month
            $holidays = PublicHoliday::where(function ($query) use ($startOfMonth, $endOfMonth) {
                $query->where('from', '<=', $endOfMonth->format('Y-m-d'))
                      ->where('to', '>=', $startOfMonth->format('Y-m-d'));
            })->get();
            
            // Create array of holiday dates
            $holidayDates = [];
            foreach ($holidays as $holiday) {
                $holidayStart = Carbon::parse($holiday->from);
                $holidayEnd = Carbon::parse($holiday->to);
                
                // Handle single day holidays
                if ($holidayStart->eq($holidayEnd)) {
                    $holidayDates[] = $holidayStart->format('Y-m-d');
                } else {
                    // Handle multi-day holidays
                    $currentHolidayDate = $holidayStart->copy();
                    while ($currentHolidayDate->lte($holidayEnd)) {
                        // Only include holiday dates within the month
                        if ($currentHolidayDate->gte($startOfMonth) && $currentHolidayDate->lte($endOfMonth)) {
                            $holidayDates[] = $currentHolidayDate->format('Y-m-d');
                        }
                        $currentHolidayDate->addDay();
                    }
                }
            }
            $holidayDates = array_unique($holidayDates);
            
            // Calculate working days excluding Saturdays, Sundays, and holidays
            $workingDays = 0;
            $currentDate = $startOfMonth->copy();
            
            while ($currentDate->lte($endOfMonth)) {
                $currentDateStr = $currentDate->format('Y-m-d');
                
                // Skip Saturday (6), Sunday (0), and holidays
                if ($currentDate->dayOfWeek !== 6 && 
                    $currentDate->dayOfWeek !== 0 && 
                    !in_array($currentDateStr, $holidayDates)) {
                    $workingDays++;
                }
                $currentDate->addDay();
            }
            $dailyWage = ($workingDays > 0) ? ($basicSalary / $workingDays) : 0;
            if ($dailyWage <= 0) {
                return; // No valid daily wage, skip fine calculation
            }

            $checkInTime = Carbon::parse($attendance->date . ' ' . $attendance->check_in);
            $fineStartTime = Carbon::parse($attendance->date . ' ' . env('FINE_START_TIME', '08:16'));
            $fine20Time    = Carbon::parse($attendance->date . ' ' . env('FINE_20_TIME', '08:30'));
            $fine30Time    = Carbon::parse($attendance->date . ' ' . env('FINE_30_TIME', '09:00'));

            $finePercent = 0;
            if ($checkInTime->gt($fineStartTime) && $checkInTime->lte($fine20Time)) {
                $finePercent = 10;
            } elseif ($checkInTime->gt($fine20Time) && $checkInTime->lte($fine30Time)) {
                $finePercent = 20;
            } elseif ($checkInTime->gt($fine30Time)) {
                $finePercent = 30;
            }

            if ($finePercent > 0) {
                $fineAmount = round(($finePercent / 100) * $dailyWage, 2);
                // Only create a fine if one does not already exist for this employee and date
                $existingFine = Fine::where('employee_id', $attendance->employee_id)
                    ->where('date', $attendance->date)
                    ->where('type', 0) // 0 = Fine
                    ->where('is_attendance_related', 1) // 1 = Attendance related fine, 0 = Non-attendance related fine
                    ->first();

                if (!$existingFine) {
                    Fine::create([
                        'employee_id' => $attendance->employee_id,
                        'date' => $attendance->date,
                        'fine_amount' => $fineAmount,
                        'type' => 0, // 0 = Fine
                        'fine_reason' => "Late check-in: {$checkInTime->format('g:i A')}, {$finePercent}% of daily wage",
                    ]);
                }
            }
        } catch (\Exception $e) {
            Log::error('Late fine calculation failed: ' . $e->getMessage());
            // Don't fail the check-in process if fine calculation fails
        }
    }
    public function _check_out(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'checkin_id' => 'required',
            // 'check_out' => 'date_format:H:i',
            'longitude' => 'required',
            'latitude' => 'required',
            // 'date' => 'required|date',
        ]);
        $currentTime = now()->format('H:i');
        $today = now()->format('Y-m-d');
        if ($validator->fails()) {
            $error = $validator->errors()->first();
    
            return response()->json([
                'message' => $error
            ], 422);
        } else {
            $validatedData = $validator->validated();
    
            $attendanceRecord = EmployeeAttendance::where([
                ['id', $validatedData['checkin_id']],
                ['check_out', null]
            ])->first();


            if (!$attendanceRecord) {
                $attendanceRecordWithCheckout = EmployeeAttendance::where('id',$validatedData['checkin_id'])->where('check_out','!=',null)->first();
                if($attendanceRecordWithCheckout){
                    $message = 'Your shift has been auto logout from the system at ' . formatTime($attendanceRecordWithCheckout->check_out);
                    $is_checkout = 1;
                }else{
                    $message = 'Check-in required before check-out. Please complete your check-in before proceeding with check-out.';
                    $is_checkout = 1;
                }
                /////////////////// ATTENDANCE LOGS ////////////////////
                $log = [
                    'employee_id' => Auth::user()->id,
                    'site_id' => null,
                    'check_in' => null,
                    'check_out' => $currentTime,
                    'date' => formatDate($today),
                    'longitude' => $validatedData['longitude'],
                    'latitude' => $validatedData['latitude'],
                    'message' => $message,
                    'error_type' => 'check-in-missed',
                    'request_type' => 'attendance'
                ];
                storeAttendanceLogs($log);
                /////////////////// ATTENDANCE LOGS END ////////////////////
                return response()->json([
                    'message' => $message,
                    'is_checkout' => $is_checkout
                ], 422);
            }

            // --- WORK FROM HOME LOGIC ---
            $leave = LeaveRequest::where('employee_id', Auth::user()->id)
                ->where('status', 1)
                ->where('from', '<=', $today)
                ->where('to', '>=', $today)
                ->with(['leavepackage.leaveType'])
                ->first();
            $isWFH = false;
            if ($leave) {
                $leaveType = $leave->leavepackage ? $leave->leavepackage->leaveType : null;
                if ($leaveType) {
                    $leaveHours = (int) $leaveType->leave_hours;
                    if ($leaveHours === 0) {
                        $isWFH = true; // Work From Home
                    }
                }
            }

            // Skip site validation for work-from-home employees
            if (!$isWFH) {
                //////////////////////
                $point = [
                    'longitude' => $validatedData['longitude'],
                    'latitude' => $validatedData['latitude'],
                ];
                
                $site = Sites::find($attendanceRecord->site_id);
                // return  $site;
                if (!$site) {
                    $message = "The site you are trying to check out from is not available now.";
                    /////////////////// ATTENDANCE LOGS ////////////////////
                    $log = [
                        'employee_id' => Auth::user()->id,
                        'site_id' => $attendanceRecord->site_id,
                        'check_in' => $attendanceRecord->check_in,
                        'check_out' => $currentTime,
                        'date' => formatDate($today),
                        'longitude' => $validatedData['longitude'],
                        'latitude' => $validatedData['latitude'],
                        'message' => $message,
                        'error_type' => 'site-not-available',
                        'request_type' => 'attendance'
                    ];
                    storeAttendanceLogs($log);
                    /////////////////// ATTENDANCE LOGS END ////////////////////
                    return response()->json([
                        'message' => $message
                    ], 422);
                }
                
                $center = [
                    'longitude' => $site->longitude,
                    'latitude' => $site->latitude,
                ];
                $radius = $site->area_radius;
                $allSites =Sites::all();
                $data = [];
                $data = [
                    'employee_location' => $point,
                    'site_location' => $center,
                    'site_radius' => $radius,
                    'all_sites' => $allSites
                ];
                
                $foundSite = null;
            
                foreach ($allSites as $allSite) {
                    $center = [
                        'longitude' => $allSite->longitude,
                        'latitude' => $allSite->latitude,
                    ];
                    $radius = $allSite->area_radius;
            
                    if (GeoLoacation::isPointWithinRadius($point, $center, $radius)) {
                        $foundSite = $allSite;
                        break;
                    }
                }
               
               

                if(!$foundSite) {
                    $message = 'You are trying to check out outside from the site. Your assigned site is: ' . $site->title ?? 'Global Site' . '.';

                    /////////////////// ATTENDANCE LOGS ////////////////////
                    $log = [
                        'employee_id' => Auth::user()->id,
                        'site_id' => $attendanceRecord->site_id,
                        'check_in' => $attendanceRecord->check_in,
                        'check_out' => $currentTime,
                        'date' => formatDate($today),
                        'longitude' => $validatedData['longitude'],
                        'latitude' => $validatedData['latitude'],
                        'message' => $message,
                        'error_type' => 'site-area-invalid',
                        'request_type' => 'attendance'
                    ];
                    storeAttendanceLogs($log);
                    /////////////////// ATTENDANCE LOGS END ////////////////////
                    return response()->json([
                        'message' => $message
                    ], 422);
                }
                
                if($foundSite && $foundSite->id !== $site->id) {
                    $message = 'You are trying to check out from the site: ' . $foundSite->title ?? 'Global Site' . '. Your assigned site is: ' . $site->title ?? 'Global Site' . '.';

                    /////////////////// ATTENDANCE LOGS ////////////////////
                    $log = [
                        'employee_id' => Auth::user()->id,
                        'site_id' => $attendanceRecord->site_id,
                        'check_in' => $attendanceRecord->check_in,
                        'check_out' => $currentTime,
                        'date' => formatDate($today),
                        'longitude' => $validatedData['longitude'],
                        'latitude' => $validatedData['latitude'],
                        'message' => $message,
                        'error_type' => 'site-area-invalid',
                        'request_type' => 'attendance'
                    ];
                    storeAttendanceLogs($log);
                    /////////////////// ATTENDANCE LOGS END ////////////////////
                    return response()->json([
                        'message' => $message
                    ], 422);
                }
            }
            
                  /////////////////////


            // $checkIn = Carbon::parse($attendanceRecord->check_in);
            // $checkOut = Carbon::parse($validatedData['check_out']);
              $checkIn = Carbon::parse($attendanceRecord->date .' ' .$attendanceRecord->check_in);
            $checkOut = Carbon::parse($today.' '.$currentTime);
            
            $totalHours = $checkOut->diffInMinutes($checkIn);
            
            if ($checkOut->lte($checkIn)) {
                $message = 'Check-out time must come after check-in time.';
                /////////////////// ATTENDANCE LOGS ////////////////////
                $log = [
                    'employee_id' => Auth::user()->id,
                    'site_id' => $attendanceRecord->site_id,
                    'check_in' => $attendanceRecord->check_in,
                    'check_out' => $currentTime,
                    'date' => formatDate($today),
                    'longitude' => $validatedData['longitude'],
                    'latitude' => $validatedData['latitude'],
                    'message' => $message,
                    'error_type' => 'check-out-time-invalid',
                    'request_type' => 'attendance'
                ];
                storeAttendanceLogs($log);
                /////////////////// ATTENDANCE LOGS END ////////////////////
                return response()->json([
                    'message' => $message
                ], 422);
            }

            $breaks = EmployeeBreak::where('emp_attendance_id',$attendanceRecord->id)->get();
            // Calculate total breaks
            $totalBreaks = 0;
            $totalWorkingHours = 0;
            if (isset($breaks)) {
                foreach ($breaks as $break) {
                        $breakInTime = Carbon::parse($break->break_in);
                        $breakOutTime = Carbon::parse($break->break_out);
                        $totalBreaks += $breakOutTime->diffInMinutes($breakInTime);
                }
            }
          // Calculate total working hours with breaks
            $totalWorkingHours = $totalHours - $totalBreaks;

            $lunchThreshold = Carbon::parse($today.' 14:00');
            $checkIn = Carbon::parse($attendanceRecord->check_in);

            // Only deduct 60 minutes if check-in is before 14:00 and check-out is after 14:00
            if ($checkIn->lt($lunchThreshold) && $checkOut->gt($lunchThreshold)) {
                $totalWorkingHours -= 60;
            }

            $attendanceRecord->update([
                'check_out' => $currentTime,
                'working_hours' => $totalWorkingHours
            ]);
    
            $message = $isWFH ? 'Check-out successful (Work From Home)' : 'success';
            ///////////////// ATTENDANCE LOGS ////////////////////
            $log = [
                'employee_id' => Auth::user()->id,
                'site_id' => $attendanceRecord->site_id,
                'check_in' => $attendanceRecord->check_in,
                'check_out' => $currentTime,
                'date' => formatDate($today),
                'longitude' => $validatedData['longitude'],
                'latitude' => $validatedData['latitude'],
                'message' => $message,
                'error_type' => 'success',
                'request_type' => 'attendance'
            ];
             /////////////////// ATTENDANCE LOGS END ////////////////////
            storeAttendanceLogs($log);
            return response()->json([
                'message' =>$message,
                'data' => $attendanceRecord
            ], 200);
        }
    }

    public function last_check_in(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'employee_id' => 'required',
        ]);
    
        if ($validator->fails()) {
            $error = $validator->errors()->first();
    
            return response()->json([
                'message' => $error
            ], 422);
        } else {
            $validatedData = $validator->validated();

            $lastcheckin = EmployeeAttendance::where('employee_id', $validatedData['employee_id'])
            ->latest()
            ->first();
        if(isset($lastcheckin->check_out)){
            return response()->json([
                'message' => 'User Check In Detail',
                // 'data' => $lastcheckin
            ], 200);
           } else{
                return response()->json([
                'message' => 'User Check In Detail',
                'data' => $lastcheckin
            ], 200);
            }
        }
    }
    public function break_in(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'emp_attendance_id' => 'required',
            'break_in'=>'date_format:H:i',
            'date' => 'required|date',
        ]);
    
        if ($validator->fails()) {
            $error = $validator->errors()->first();
    
            return response()->json([
                'message' => $error
            ], 422);
        } else {

            $validatedData = $validator->validated();

            $data = [
                'emp_attendance_id' => $validatedData['emp_attendance_id'],
                'break_in' => $validatedData['break_in'],
                'date' => $validatedData['date'],
            ];
  
            $breakin = EmployeeBreak::create($data);

            return response()->json([
                'message' => 'Break In successfully',
                'data' => $breakin
            ], 200);
        } 
    }
    
    public function last_break_in(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'emp_attendance_id' => 'required',
        ]);
    
        if ($validator->fails()) {
            $error = $validator->errors()->first();
    
            return response()->json([
                'message' => $error
            ], 422);
        } else {

            $validatedData = $validator->validated();

            $lastbreakin = EmployeeBreak::where('emp_attendance_id', $validatedData['emp_attendance_id'])
            ->latest()
            ->first();
            
            if ($lastbreakin) {
                
            if ($lastbreakin->break_out) {
            return response()->json([
            'message' => 'Break In Detail',
            ], 200);
            } else {
                return response()->json([
                'message' => 'Break In Detail',
                'data' => $lastbreakin
                ], 200);
                }
            } else {
                return response()->json([
                'message' => 'No breaks found.',
                ], 200);
            }
                
            
        }
    }

    public function break_out(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'breakin_id' => 'required',
            'break_out'=>'date_format:H:i',
        ]);
    
        if ($validator->fails()) {
            $error = $validator->errors()->first();
            return response()->json([
                'message' => $error
            ], 422);
        } else {

            $validatedData = $validator->validated();

            $attendanceRecord = EmployeeBreak::find($validatedData['breakin_id']);

            $attendanceRecord->update([
                'break_out' => $validatedData['break_out'],
            ]);
    
            return response()->json([
                'message' => 'break Out successfully',
                'data' => $attendanceRecord
            ], 200);
        } 
    }

    public function updateWorkingHoursByDate()
    {
        $attendances = EmployeeAttendance::whereYear('date', 2025)
                                        ->whereMonth('date', 3)
                                        ->get();

        $groupedAttendances = $attendances->groupBy(function($attendance) {
            return $attendance->employee_id . '-' . $attendance->date;
        });

        foreach ($groupedAttendances as $key => $attendanceGroup) {
            $attendance = $attendanceGroup->first();
            if($attendance->working_hours > 0){
                $attendance->working_hours += 60;
                $attendance->save();
            }
           
        }

        return response()->json([
            'message' => 'Working hours updated successfully for March 2025',
            'updated_count' => count($groupedAttendances),
        ], 200);
    }

    // public function checkin_history(Request $request)
    // {
    //     $validator = Validator::make($request->all(), [
    //         'employee_id' => 'required',
    //         'start_date' => 'required|date',
    //         'emp_attendance_id' => 'required',
    //         // 'end_date' => 'required|date'
    //     ]);

    //     if ($validator->fails()) {
    //         $error = $validator->errors()->first();

    //         return response()->json([
    //             'message' => $error
    //         ], 422);
    //     } else {
    //         $validatedData = $validator->validated();

    //         // $empattendance = EmployeeAttendance::where('employee_id', $validatedData['employee_id'])
    //         //     ->whereBetween('date', [$validatedData['start_date'], $validatedData['end_date']])
    //         //     ->get();
    //         //       $empattendance = EmployeeAttendance::where('employee_id', $validatedData['employee_id'])
    //         // ->where('date',$validatedData['start_date'])
    //         // ->with('breaks') 
    //         // ->get();
    //         $empattendance = EmployeeAttendance::where('employee_id', $validatedData['employee_id'])
    //         ->where('date', $validatedData['start_date'])
    //         ->whereHas('breaks', function ($query) use ($validatedData) {
    //             $query->where('emp_attendance_id', $validatedData['emp_attendance_id']);
    //         })
    //         ->with('breaks')
    //         ->get();



    //         if ($empattendance->isNotEmpty()) {
    //             return response([
    //                 'message' => 'User Check In History',
    //                 'data' => $empattendance
    //             ], 200);
    //         } else {
    //             return response([
    //                 'message' => 'No records found for the specified date range'
    //             ], 200);
    //         }
    //     }
    // }
    public function checkin_history(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'employee_id' => 'required',
            'start_date' => 'required|date',
            'emp_attendance_id' => 'required',
            // 'end_date' => 'required|date'
        ]);
    
        if ($validator->fails()) {
            $error = $validator->errors()->first();
    
            return response()->json([
                'message' => $error
            ], 422);
        } else {
            $validatedData = $validator->validated();
    
            // $empattendance = EmployeeAttendance::where('employee_id', $validatedData['employee_id'])
            //     ->where('date', $validatedData['start_date'])
            //     ->where(function ($query) use ($validatedData) {
            //         $query->whereHas('breaks', function ($innerQuery) use ($validatedData) {
            //             $innerQuery->where('emp_attendance_id', $validatedData['emp_attendance_id']);
            //         })
            //         ->orWhereDoesntHave('breaks');
            //     })
            //     ->with('breaks')
            //     ->latest('id')
            //     ->get();
            $empattendance = EmployeeAttendance::where('employee_id', $validatedData['employee_id'])
                ->where('date', $validatedData['start_date'])
                ->where(function ($query) use ($validatedData) {
                    $query->whereHas('breaks', function ($innerQuery) use ($validatedData) {
                        $innerQuery->where('emp_attendance_id', $validatedData['emp_attendance_id']);
                    })
                    ->orWhereDoesntHave('breaks');
                })
                ->with([
                    'breaks' => function ($query) use ($validatedData) {
                        $query->where('emp_attendance_id', $validatedData['emp_attendance_id']);
                    },
                    'sites' => function ($query) {
                        $query->select('id', 'title', 'street_address');
                    }
                ])
                ->latest('id')
                ->first();

            
            // dd($empattendance);
            if ($empattendance) {
                return response([
                    'message' => 'User Check In History',
                    'data' => $empattendance
                ], 200);
            } else {
                return response([
                    'message' => 'No records found for the specified date range'
                ], 200);
            }
        }
    }

    public function attendance_list(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'employee_id' => 'required',
        ]);

        if ($validator->fails()) {
            $error = $validator->errors()->first();

            return response()->json([
                'message' => $error
            ], 422);
        } else {
            $validatedData = $validator->validated();

            $firstDayOfPreviousMonth = Carbon::now()->subMonth()->startOfMonth();
            $lastDayOfPreviousMonth = Carbon::now()->subMonth()->endOfMonth();

            $empattendance = EmployeeAttendance::where('employee_id', $validatedData['employee_id'])
                ->whereBetween('date', [$firstDayOfPreviousMonth, $lastDayOfPreviousMonth])
                ->get();

            if ($empattendance->count() > 0) {
                return response()->json([
                    'message' => 'User Attendance List for the previous month',
                    'data' => $empattendance
                ], 200);
            } else {
                return response()->json([
                    'message' => 'No Record Found for the previous month.'
                ], 404);
            }
        }
    }
        ////////////////Attendance Api///////////////

        ///////////////// CRON JOB //////////////////
   public function autoCheckout(){
        $date_from = Carbon::now()->subDays(2)->format('Y-m-d');
        $current_date = Carbon::now()->format('Y-m-d');
        $current_time = Carbon::now()->format('H:i:s');
        $employee_attendances = EmployeeAttendance::whereNotNull('check_in')->whereNull('check_out')
        ->where('date',">=",$date_from)->get();
    
        Log::info(date('d-m-y H:i:s') . " Employees who did not checkout " . $employee_attendances);
        $update_attendance = $delete_attendance = $check = 0;
        $all_employee_id = [];
        foreach($employee_attendances as $attendance){
            // New logic for half-day leave
            $leave = LeaveRequest::where('employee_id', $attendance->employee_id)
                ->whereIn('status', [1, 4]) // Approved
                ->whereDate('from', '<=', $attendance->date)
                ->whereDate('to', '>=', $attendance->date)
                ->whereHas('leavepackage.leaveType', function ($query) {
                    $query->where('leave_hours', 4);
                })
                ->first();

            if ($leave) {
                // Half-day leave found.
                $check_in_datetime = Carbon::parse($attendance->date . ' ' . $attendance->check_in);
                $auto_checkout_datetime = $check_in_datetime->copy()->addHours(4);
                $current_datetime = Carbon::now();

                if ($current_datetime >= $auto_checkout_datetime) {
                  EmployeeAttendance::where('id', $attendance->id)
                        ->update([
                            'check_out' => $auto_checkout_datetime->format('H:i:s'),
                            'checkout_type' => 1 // System checkout
                        ]);
                        $attendance_update =  EmployeeAttendance::where('id', $attendance->id)->first();
                    // Calculate working hours after updating checkout time
                    $total_working_minutes = $this->calculateWorkingHours($attendance_update, $attendance_update->check_in, $auto_checkout_datetime->format('H:i:s'));
                    
                    // Update working hours
                    EmployeeAttendance::where('id', $attendance->id)
                        ->update(['working_hours' => $total_working_minutes]);
                        
                    $update_attendance++;
                    $check = 1;
                    if(!in_array($attendance->employee_id, $all_employee_id)) {
                        $all_employee_id[] = $attendance->employee_id;
                    }

                    continue; // Skip the rest of the logic for this attendance.
                }
            }
            
            // Get all rosters for this employee on this date
            $rosters = RosterAssign::join('roster_templates','roster_assigns.roster_template_id','roster_templates.id')
                            ->where('roster_assigns.assign_to', $attendance->employee_id)
                            ->where('roster_assigns.schedule_date', $attendance->date)
                            ->select('roster_assigns.*', 'roster_templates.start_time', 'roster_templates.end_time')
                            ->orderBy('roster_templates.start_time')
                            ->get();
    
            if($rosters->count() > 0){
                // Get the check-in time of this attendance
                $check_in_time = $attendance->check_in;
                $check_in_datetime = new \DateTime($check_in_time);
                
                // Find which roster this check-in belongs to
                $matched_roster = null;
                
                foreach($rosters as $roster){
                    $roster_start = new \DateTime($roster->start_time);
                    $roster_end = new \DateTime($roster->end_time);
                    
                    // Add a buffer before the start time (for early check-ins)
                    $buffer_before = clone $roster_start;
                    $buffer_before->modify('-60 minutes'); // 1-hour buffer
                    
                    // If roster ends after midnight, adjust for comparison
                    if($roster_start > $roster_end) {
                        $roster_end->modify('+1 day');
                    }
                    
                    // Check if this attendance's check-in time falls within this roster's timeframe (with buffer)
                    if($check_in_datetime >= $buffer_before && $check_in_datetime <= $roster_end) {
                        $matched_roster = $roster;
                        break; // Found the matching roster
                    }
                }
                
                // If no exact match found but there are rosters, use the closest one by start time
                if(!$matched_roster && $rosters->count() > 0) {
                    $closest_diff = PHP_INT_MAX;
                    
                    foreach($rosters as $roster) {
                        $roster_start = new \DateTime($roster->start_time);
                        $diff = abs($check_in_datetime->getTimestamp() - $roster_start->getTimestamp());
                        
                        if($diff < $closest_diff) {
                            $closest_diff = $diff;
                            $matched_roster = $roster;
                        }
                    }
                }
                
                if($matched_roster) {
                    if($attendance->date < $current_date){
                        // For past dates, use the matched roster's end time for checkout
                        EmployeeAttendance::where('id', $attendance->id)
                            ->update([
                                'check_out' => formatTime($matched_roster->end_time),
                                'checkout_type' => 1
                            ]);
                        
                        // Get updated attendance record
                        $attendance_update = EmployeeAttendance::where('id', $attendance->id)->first();
                        
                        // Calculate working hours after updating checkout time
                        $total_working_minutes = $this->calculateWorkingHours($attendance_update, $attendance_update->check_in, $attendance_update->check_out);
                        
                        // Update working hours
                        EmployeeAttendance::where('id', $attendance->id)
                            ->update(['working_hours' => $total_working_minutes]);
                            
                        $update_attendance += 1;
                        $check = 1;
                    } elseif($attendance->date == $current_date) {

                        // For current date, only auto-checkout if current time is past the roster end time
                        $roster_end_time = new \DateTime($matched_roster->end_time);
                        $currentTimeObj = new \DateTime($current_time);
                        if($currentTimeObj >= $roster_end_time) {
                            $checkout_time = (new \DateTime($matched_roster->end_time))->modify('-'.env('CHECKOUT_TIME_BUFFER', 20).' minutes')->format('H:i');
                            
                            EmployeeAttendance::where('id', $attendance->id)
                                ->update([
                                    'check_out' => formatTime($checkout_time),
                                    'checkout_type' => 1
                                ]);
                            
                            // Get updated attendance record
                            $attendance_update = EmployeeAttendance::where('id', $attendance->id)->first();
                            
                            // Calculate working hours after updating checkout time
                            $total_working_minutes = $this->calculateWorkingHours($attendance_update, $attendance_update->check_in, $attendance_update->check_out);
                            
                            // Update working hours
                            EmployeeAttendance::where('id', $attendance->id)
                                ->update(['working_hours' => $total_working_minutes]);
                                
                            $update_attendance += 1;
                            $check = 1;
                        }
                    }
                    
                    if(!in_array($attendance->employee_id, $all_employee_id)) {
                        $all_employee_id[] = $attendance->employee_id;
                    }
                } else {
                    // No appropriate roster found for this attendance
                    EmployeeAttendance::where('id', $attendance->id)->delete();
                    $delete_attendance += 1;
                    $check = 1;
                }
            } else {
                // No rosters found for this employee on this date
                EmployeeAttendance::where('id', $attendance->id)->delete();
                $delete_attendance += 1;
                $check = 1;
            }
        }
        
        $employee_details = EmpCompanyDetails::whereIn('id', $all_employee_id)->select('id','employee_email')->get();
        Log::info(date('d-m-y H:i:s') . " Auto Checkout API Run successfully. Here are the employees details " . $employee_details);
        
        $data['total_updated_attendance'] = $update_attendance;
        $data['total_deleted_attendance'] = $delete_attendance;
        
        if($check){
            return response()->json([
                'message' => 'All employees check-out successfully',
                'data' => $data
            ], 200);
        } else {
            return response()->json([
                'message' => 'Nothing to update and delete',
                'data' => $data
            ], 404);
        }
    }


    public function attendanceDownload(Request $request)
    {
           $query = EmployeeAttendance::with(['EmpPersonalDetails','EmpCompanyDetails','Sites','empTeamsMembers']);

           if($request->filled('filter'))
           {
              $query = $this->filter(json_decode($request->filter, true),$query);
           }
            
           $attendances = $query->get();
          
           $cols = DB::table('attendance_list_change_cols')->pluck('col')->toArray();

           $columns = [
               'Employee',
               'Type',
               'Teams',
               'Site',
               'Check In',
               'Check Out',
               'Working Hours',
               'Date'
           ];

            $col_list[] = "Employee Id";
            $col_list[] = "External Id";
            $indexs = [];
            $i = 2;
            foreach ($columns as $column) {
              if (in_array($column, $cols))
              {
                $col_list[] = $column;
              }
              else
              {
                $indexs[] = $i;
              }
              $i++;
            }
  
            $csvData[] = $col_list;
  
            $data_list  = [];
          
       
            $user_meta = '';
            foreach ($attendances as $attendance) {
                $team_list = "";
              foreach($attendance->empTeamsMembers as $et)
              {
                $team_list = $et->empTeamsList->title.",";
              }
              
               $user_meta =  DB::table('user_meta')->where('emp_id',$attendance->employee_id)->where('option','employee_payroll_or_external_id')->first();
               $data_list = [ 
                    $attendance->employee_id,
                    $user_meta ? $user_meta->value : '',
                    $attendance->emppersonaldetails->first_name.' '.$attendance->emppersonaldetails->middle_name.' '.$attendance->emppersonaldetails->last_name,
                    $attendance->empcompanydetails->user_type == 0 ?  'Internal' : 'External',
                    $team_list  ? rtrim($team_list,',') : "",
                    $attendance->sites->title ?? 'Global Site',
                    $this->AM_PM($attendance->check_in),
                    $attendance->check_out ? $this->AM_PM($attendance->check_out) : $attendance->check_out,
                    $this->minutesToHours($attendance->working_hours),
                    date("d-m-Y", strtotime($attendance->date))
                ];

                $csvData[] = array_values(array_diff_key($data_list, array_flip($indexs)));
            }

            $headers = [
                'Content-Type' => 'text/csv',
                'Content-Disposition' => 'attachment; filename="attendance.csv"', 
            ];

            
            $callback = function () use ($csvData) {
                $file = fopen('php://output', 'w');
                foreach ($csvData as $row) {
                    fputcsv($file, $row);
                }
                fclose($file);
            };

            return Response::stream($callback, 200, $headers);
    }


    function AM_PM($TIME)
    {
        $time = Carbon::parse($TIME);
        $time = $time->format('g:i A');
        $time = str_replace(' ', '',$time);
        $time = str_replace(':00', '',$time);
        return $time;
    }


    function minutesToHours($minutes) {
        $isNegative = $minutes < 0;
        $minutes = $minutes == '' ?  0 :   abs($minutes);
    
        $hours = floor($minutes / 60);
        $remainingMinutes = $minutes % 60;
    
        $result = ($isNegative ? '-' : '') . $hours . 'h:' . ($remainingMinutes < 10 ? '0' : '') . $remainingMinutes . 'm';
    
        return $result;
    }

    public function changeCol(Request $request)
    {
      $arr = [];
      $col = $request->col;
      if($col){
        DB::table('attendance_list_change_cols')->delete();
        for($i = 0; $i < count($col); $i++)
        {
          $arr[$i]['col'] = $col[$i];
        }
        AttendanceListChangeCol::insert($arr);
      }
      return back();
    }
    
    public function calculateWorkingHours($employee_attendance, $check_out, $current_time){
        $checkIn = Carbon::parse($employee_attendance->date . ' ' . $employee_attendance->check_in);
        $checkOut = Carbon::parse($employee_attendance->date . ' ' . $employee_attendance->check_out);
        $totalHours = $checkOut->diffInMinutes($checkIn);
    
        $breaks = EmployeeBreak::where('emp_attendance_id', $employee_attendance->id)->get();
        // Calculate total breaks
        $totalBreaks = 0;
        $totalWorkingHours = 0;
        if (isset($breaks)) {
            foreach ($breaks as $break) {
                if(is_null($break->break_out) && $current_time > $check_out){
                    $break_update = EmployeeBreak::where('id', $break->id)->update(['break_out' => $check_out]);
                    $breakOutTime = Carbon::parse($check_out);
                }else{
                    $breakOutTime = Carbon::parse($break->break_out);
                }
                $breakInTime = Carbon::parse($break->break_in);
                $totalBreaks += $breakOutTime->diffInMinutes($breakInTime);
            }
        }
        // Calculate total working hours with breaks
        $totalWorkingHours = $totalHours - $totalBreaks;
        
        // Check if both check-in and check-out times are after 14:00 (2:00 PM)
        if ($checkIn->format('H:i') >= '14:00' && $checkOut->format('H:i') > '14:00') {
            // Don't subtract anything when both are after 14:00
        } else {
            $totalWorkingHours = $totalWorkingHours - 60; // Subtract 60 minutes in all other cases
        }
        
        return $totalWorkingHours;
    }
    public function calculateHours(Request $request)
    {
        $request->validate([
            'employee_id' => 'required|integer',
            'year' => 'required|integer|min:2000|max:2999',
            'month' => 'required|integer|min:1|max:12'
        ]);

        $employeeId = $request->employee_id;
        $year = $request->year;
        $month = $request->month;

        $startDate = Carbon::createFromDate($year, $month, 1)->startOfDay();
        $endDate = $startDate->copy()->endOfMonth()->endOfDay();

        $attendances = EmployeeAttendance::where('employee_id', $employeeId)
                                 ->whereBetween('date', [$startDate, $endDate])
                                 ->get();

        $totalHours = 0;
        foreach ($attendances as $attendance) {
            $checkIn = Carbon::parse($attendance->check_in);
            $checkOut = Carbon::parse($attendance->check_out);

            $hours = $checkOut->diffInHours($checkIn);
            $totalHours += $hours;
        }
        return response()->json([
            'total_hours' => $totalHours,
            'message' => 'Total hours calculated successfully.'
        ]);
    }

    public function attendanceExport(Request $request){
        $request->validate([
            'to' => 'required|date',
            'from' => 'required|date|before_or_equal:to',
            'empid' => 'required',
            'site_id' => 'nullable'
        ]);
    
        $toDate = $request->input('to');
        $fromDate = $request->input('from');
        $siteid = $request->input('site_id');
        $empid = $request->input('empid');
    
        // Get attendance records
        $attendanceQuery = EmployeeAttendance::where('employee_id', $empid)
                   ->whereBetween('date', [$fromDate, $toDate])
                   ->with(['RosterAssign.RosterTemplate', 'sites', 'breaks']);
                   
        // Get overtime records
        $overtimeQuery = Overtime::where('employee_id', $empid)
                    ->whereBetween('date', [$fromDate, $toDate])
                    ->where('status', 2) // Only approved overtime records
                    ->with(['sites']);
                    
        // Get leave records
        $leaveQuery = LeaveRequest::where('employee_id', $empid)
                    ->where(function($query) use ($fromDate, $toDate) {
                        $query->whereBetween('from', [$fromDate, $toDate])
                              ->orWhereBetween('to', [$fromDate, $toDate])
                              ->orWhere(function($q) use ($fromDate, $toDate) {
                                  $q->where('from', '<=', $fromDate)
                                    ->where('to', '>=', $toDate);
                              });
                    })
                    ->with(['leavepackage']);
                    
        // Get roster records for required hours calculation
        $rosterQuery = RosterAssign::join('roster_templates', 'roster_assigns.roster_template_id', '=', 'roster_templates.id')
                    ->where('roster_assigns.assign_to', $empid)
                    ->whereBetween('roster_assigns.schedule_date', [$fromDate, $toDate])
                    ->select('roster_assigns.schedule_date', 'roster_templates.working_hours');
    
        if ($siteid) {
            $attendanceQuery = $attendanceQuery->where('site_id', $siteid);
            $overtimeQuery = $overtimeQuery->where('site_id', $siteid);
        }
    
        $attendances = $attendanceQuery->orderBy('date', 'desc')->get();
        $overtimes = $overtimeQuery->orderBy('date', 'desc')->get();
        $leaves = $leaveQuery->orderBy('from', 'desc')->get();
        $rosters = $rosterQuery->get();
        
        $employee = EmpPersonalDetails::where('emp_id', $empid)->select('emp_id','first_name','middle_name','last_name')->first();
    
        // Construct file name using employee details
        $fullName = trim($employee->first_name . ' ' . ($employee->middle_name ?? '') . ' ' . $employee->last_name);
        $fileName = $employee->emp_id . '_' . str_replace(' ', '_', $fullName) . '_attendance.csv';

        $headers = [
            "Content-type" => "text/csv",
            "Content-Disposition" => "attachment; filename=\"$fileName\"",
            "Pragma" => "no-cache",
            "Cache-Control" => "must-revalidate, post-check=0, pre-check=0",
            "Expires" => "0"
        ];
    
        $columns = ['Date', 'Day', 'Site', 'Check-In/Check-Out', 'Break-In/Break-Out', 'Total Required Hours', 'Spent Hours', 'Difference (Hours and Minutes)', 'Overtime', 'Type', 'Status'];
    
        $callback = function() use ($attendances, $overtimes, $leaves, $rosters, $columns) {
            $file = fopen('php://output', 'w');
            fputcsv($file, $columns);
        
            $totalEmployeeWorkingHours = 0;
            $totalOvertime = 0;
            $totalRequiredHours = 0;
            
            // Create a map of roster hours by date
            $rosterHoursByDate = [];
            foreach ($rosters as $roster) {
                $rosterHoursByDate[$roster->schedule_date] = $roster->working_hours;
                $totalRequiredHours += $roster->working_hours;
            }
            
            // Calculate total approved leave hours (8 hours per approved leave)
            $totalApprovedLeaveHours = 0;
            $approvedLeavesByDate = [];
            foreach ($leaves as $leave) {
                if (in_array($leave->status, [1, 4])) { // Consider both approved (1) and approved as unpaid (4) leaves
                    $currentDate = Carbon::parse($leave->from);
                    $endDate = Carbon::parse($leave->to);
                    
                    while ($currentDate <= $endDate) {
                        $dateStr = $currentDate->format('Y-m-d');
                        $approvedLeavesByDate[$dateStr] = 480; // 8 hours = 480 minutes
                        $totalApprovedLeaveHours += 480;
                        $currentDate->addDay();
                    }
                }
            }
            
            // Process attendance records
            foreach ($attendances as $attendance) {
                $checkInOut = formatTime($attendance->check_in, 'g:i A') . ' - ' . formatTime($attendance->check_out, 'g:i A');
                $breakInOut = collect($attendance->breaks)->map(function ($b) {
                    return formatTime($b->break_in, 'g:i A') . ' - ' . formatTime($b->break_out, 'g:i A');
                })->join(', ');
        
                $rosterWorkingHours = $rosterHoursByDate[$attendance->date] ?? 0;
                $employeeWorkingHours = $attendance->working_hours ?? 0;
                $differenceInMinutes = $employeeWorkingHours - $rosterWorkingHours;
        
                $totalEmployeeWorkingHours += $employeeWorkingHours;
                $overtimeMinutes = max(0, $differenceInMinutes);
                $totalOvertime += $overtimeMinutes;
        
                fputcsv($file, [
                    formatDate($attendance->date),
                    date('l', strtotime($attendance->date)),
                    $attendance->sites->title ?? 'Global Site',
                    $checkInOut,
                    $breakInOut,
                    $this->convertMinutesToHoursMinutes($rosterWorkingHours),
                    $this->convertMinutesToHoursMinutes($employeeWorkingHours),
                    $this->convertMinutesToHoursMinutes($differenceInMinutes),
                    $this->convertMinutesToHoursMinutes($overtimeMinutes),
                    'Regular',
                    'Present'
                ]);
            }
            
            // Process overtime records
            foreach ($overtimes as $overtime) {
                $checkInOut = formatTime($overtime->check_in, 'g:i A') . ' - ' . formatTime($overtime->check_out, 'g:i A');
                $overtimeHours = $this->calculateOvertimeHours($overtime->check_in, $overtime->check_out);
                
                fputcsv($file, [
                    formatDate($overtime->date),
                    date('l', strtotime($overtime->date)),
                    $overtime->sites->title ?? 'Global Site',
                    $checkInOut,
                    '', // No breaks for overtime
                    '0:00', // No required hours for overtime
                    $this->convertMinutesToHoursMinutes($overtimeHours),
                    $this->convertMinutesToHoursMinutes($overtimeHours),
                    $this->convertMinutesToHoursMinutes($overtimeHours),
                    'Overtime',
                    'Present'
                ]);
                
                $totalOvertime += $overtimeHours;
            }
            
            // Process leave records
            foreach ($leaves as $leave) {
                $leaveType = $leave->leavepackage ? $leave->leavepackage->title : 'Unknown';
                $status = $this->getLeaveStatus($leave->status);
                
                $timeRange = $leave->start_time . ' - ' . $leave->end_time;
                $requiredHours = $rosterHoursByDate[$leave->from] ?? 0;
                
                fputcsv($file, [
                    formatDate($leave->from),
                    date('l', strtotime($leave->from)),
                    'N/A',
                    $timeRange,
                    '',
                    $this->convertMinutesToHoursMinutes($requiredHours),
                    '0:00',
                    $this->convertMinutesToHoursMinutes(-480), // 8 hours = 480 minutes
                    '0:00',
                    'Leave - ' . $leaveType,
                    $status
                ]);
            }
        
            // Calculate total hours including overtime
            $totalHours = $totalEmployeeWorkingHours + $totalOvertime;
            
            // Calculate total required hours after deducting approved leaves
            $totalRequiredHours -= $totalApprovedLeaveHours;
        
            // Add total rows
            fputcsv($file, [
                '', 
                '', 
                'Total Required Hours:', 
                '', 
                '', 
                $this->convertMinutesToHoursMinutes($totalRequiredHours),
                '', 
                '', 
                '',
                '',
                ''
            ]);
            
            fputcsv($file, [
                '', 
                '', 
                'Total Regular Hours:', 
                '', 
                '', 
                '', 
                $this->convertMinutesToHoursMinutes($totalEmployeeWorkingHours),
                '', 
                '',
                '',
                ''
            ]);
            
            fputcsv($file, [
                '', 
                '', 
                'Total Overtime Hours:', 
                '', 
                '', 
                '', 
                $this->convertMinutesToHoursMinutes($totalOvertime),
                '', 
                '',
                '',
                ''
            ]);
            
            fputcsv($file, [
                '', 
                '', 
                'Total Hours (Regular + Overtime):', 
                '', 
                '', 
                '', 
                $this->convertMinutesToHoursMinutes($totalHours),
                '', 
                '',
                '',
                ''
            ]);
            
            fputcsv($file, [
                '', 
                '', 
                'Total Approved Leave Hours:', 
                '', 
                '', 
                $this->convertMinutesToHoursMinutes($totalApprovedLeaveHours),
                '', 
                '', 
                '',
                '',
                ''
            ]);
        
            fclose($file);
        };
        
        return response()->stream($callback, 200, $headers);
    }
    
    /**
     * Calculate overtime hours from check-in and check-out times
     */
    private function calculateOvertimeHours($checkIn, $checkOut) {
        $checkInTime = Carbon::parse($checkIn);
        $checkOutTime = Carbon::parse($checkOut);
        return $checkOutTime->diffInMinutes($checkInTime);
    }
    
    /**
     * Converts minutes to hours and minutes format.
     */
    private function convertMinutesToHoursMinutes($minutes) {
        $hours = intdiv($minutes, 60);
        $minutes = $minutes % 60;
        return sprintf("%d:%02d", $hours, $minutes);
    }
    
    /**
     * Get leave status text based on status code
     */
    private function getLeaveStatus($status) {
        switch ($status) {
            case 0:
                return 'Pending';
            case 1:
                return 'Approved';
            case 2:
                return 'Rejected';
            default:
                return 'Unknown';
        }
    }

    /**
     * Update working hours for all employees from a given date onwards
     * This method calculates working hours by:
     * 1. Taking check_in and check_out times from employee_attendances table
     * 2. Subtracting employee breaks from employee_breaks table
     * 3. Subtracting 60 minutes for lunch break
     */
    public function calculateAndUpdateWorkingHours(Request $request)
    {
        $request->validate([
            'date' => 'required|date|before_or_equal:today',
        ]);
        $startDate = $request->date;
        $today = now()->format('Y-m-d');

        try {
            // Get all attendance records from the given date onwards
            $attendances = EmployeeAttendance::where('date', '>=', $startDate)
                ->whereNotNull('check_in')
                ->whereNotNull('check_out')
                ->with('breaks')
                ->get();

            $updatedCount = 0;
            $errors = [];

            foreach ($attendances as $attendance) {
                try {
                    // Parse check-in and check-out times
                    $checkIn = Carbon::parse($attendance->check_in);
                    $checkOut = Carbon::parse($attendance->check_out);
                    // Calculate total working minutes
                    $totalWorkingMinutes = $checkOut->diffInMinutes($checkIn);

                    // Subtract employee breaks
                    $breakMinutes = 0;
                    if ($attendance->breaks && $attendance->breaks->count() > 0) {
                        foreach ($attendance->breaks as $break) {
                            if ($break->break_in && $break->break_out) {
                                $breakStart = Carbon::parse($break->break_in);
                                $breakEnd = Carbon::parse($break->break_out);
                                $breakMinutes += $breakEnd->diffInMinutes($breakStart);
                            }
                        }
                    }

                    // Subtract lunch break (60 minutes)
                    $lunchBreakMinutes = 60;
                    if ($checkIn->format('H:i') >= '14:00' && $checkOut->format('H:i') >= '14:00') {
                        // dd($checkIn->format('H:i'),$checkOut->format('H:i'),'mursleen' );
                        $lunchBreakMinutes = 0;
                    }elseif($checkIn->format('H:i') < '14:00' && $checkOut->format('H:i') < '14:00'){
                        // dd($checkIn->format('H:i'),$checkOut->format('H:i'),'azam' );
                        $lunchBreakMinutes = 0;
                    }

                    // Calculate final working hours
                    $finalWorkingMinutes = $totalWorkingMinutes - $breakMinutes - $lunchBreakMinutes;

                    // Ensure working hours is not negative
                    $finalWorkingMinutes = max(0, $finalWorkingMinutes);
                    

                    // Update the attendance record
                    $attendance->update([
                        'working_hours' => $finalWorkingMinutes
                    ]);
                    $updatedCount++;

                } catch (\Exception $e) {
                    $errors[] = [
                        'attendance_id' => $attendance->id,
                        'employee_id' => $attendance->employee_id,
                        'date' => $attendance->date,
                        'error' => $e->getMessage()
                    ];
                }
            }

            $response = [
                'message' => 'Working hours updated successfully',
                'total_attendances_processed' => $attendances->count(),
                'successfully_updated' => $updatedCount,
                'errors' => $errors
            ];

            if (count($errors) > 0) {
                $response['message'] = 'Working hours updated with some errors';
                return response()->json($response, 207); // 207 Multi-Status
            }

            return response()->json($response, 200);

        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Error updating working hours',
                'error' => $e->getMessage()
            ], 500);
        }
    }

}
