<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Sites;
use App\Models\EmpCompanyDetails;
use App\Models\SiteHistory;
use App\Models\EmpPersonalDetails;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;


class SiteController extends Controller
{
    public function __construct()
    {
        $this->middleware('Permissions:Sites Maintain')->only(['create', 'edit']);
    }
  
    public function index(Request $request)
    {
        $query = Sites::query();
        $query->where('del','0');

        if($request->filled('filter'))
        {
          $query =  $this->filter(json_decode($request->filter, true),$query);
        }
        $query->with('EmpPersonalDetails');
  
        $count = $query->count();
        $start = $request->input('from', 0);
        $query->offset($start)->limit(10);
        $query->orderBy('id', 'desc');
        $query_result = $query->get();

        $data =  [
            'sites' => $query_result,
        ];
        $array_filter =  json_decode($request->filter, true);
        $filters = [
          'first_name' => $array_filter['first_name'] ?? '', 
          'middle_name' => $array_filter['middle_name'] ?? '',
          'last_name' => $array_filter['first_name'] ?? '',
          'employee_email' => $array_filter['last_name'] ?? '',
          'compeleted' => $array_filter['first_name'] ?? '',
          'status' => $array_filter['compeleted'] ?? '',
          'access_role' => $array_filter['access_role'] ?? '',
        ];

        return response()->json([
            'message' => 'Get Projects List Successfully',
            'data' => $data,
            'count' => $count,
            'filters' => $filters
        ],200);


    }

    public function filter($filters,$query)
    {
     
        foreach ($filters as $filterName => $filterValue) {
              if ($filterValue != null ||  $filterValue !="" ) {
                  switch ($filterName) {
                  case 'company':
                    $query->whereHas('company', function ($subquery) use ($filterValue) {
                    $subquery->where('fname', 'like', '%' . $filterValue . '%');
                  });
                  break;
                  case 'first_name':
                    $query->whereHas('emppersonaldetails', function ($subquery) use ($filterValue) {
                        $subquery->where('first_name', 'like', '%' . $filterValue . '%');
                    });
                  break;
                  case 'title':
                      $query->where('title', 'like', '%' . $filterValue . '%');
                  break;           
                  case 'area_radius':
                    $query->where('area_radius', 'like', '%' . $filterValue . '%');
                  break; 
                  case 'active':
                    $query->where('active', 'like', '%' . $filterValue . '%');
                  break;                  
                  }
              }
        }

        return $query;
    } 

    public function create(Request $request)
    {
        $get_employes = EmpCompanyDetails::with('EmpPersonalDetails')->where('del',0)->where('compeleted',1)->where('status',1)->get();
        $get_companies =[];
        return view('Sites.create',compact('get_employes','get_companies'));
    }

    public function store(Request $request)
    {

        $validator = Validator::make($request->all(), [
            'title' => 'required',
            'state' => 'required',
            'external_id' => 'nullable',
            'supervisor' => 'required',
            // 'company_client' => 'required',
            'description' => 'nullable',
            'address' => 'required',
            'latitude' => 'required',
            'longitude' => 'required',
            'area_radius' => 'required',
            'project' => 'nullable'
        ]);

        

        if ($validator->fails()) {
            $error = $validator->errors()->first();
            return response()->json([
                'message' => $error
            ], 422);
        }
        try {
            
            $validatedData = $validator->validated();
            $Data = Sites::create([
                'title' => $validatedData['title'],
                'site_state' => $validatedData['state'],
                'external_id' => $validatedData['external_id'],
                'supervisor_id' => $validatedData['supervisor'],
                'company_client' => $validatedData['company_client']??1,
                'description' => $validatedData['description'],
                'street_address' => $validatedData['address'],
                'latitude' => $validatedData['latitude'],
                'longitude' => $validatedData['longitude'],
                'area_radius' => $validatedData['area_radius'],   
                'created_by' => Auth::user()->id
            ]);
            session()->flash('success', 'Site Created Successfully');
            return response()->json(['message' => 'Site Created Successfully','Data' => $Data], 200);
        } catch (QueryException $exception) {
            $errorMessage = $exception->getMessage();
            preg_match("/'([^']+)'/", $errorMessage, $matches);
            
            $errorField = count($matches) > 1 ? $matches[1] : 'unknown';
            session()->flash('error', 'The ' . $errorField . ' field is required.');
            return response()->json(['message' => 'The ' . $errorField . ' field is required.'], 400);
        }
    }


    public function edit(Sites $site)
    {
       
        $get_employes = EmpCompanyDetails::with('EmpPersonalDetails')->where('del',0)->where('compeleted',1)->get();
      
        return view('Sites.edit', compact('site','get_employes'));
    }

    public function update(Request $request, $id)
    {

        $validator = Validator::make($request->all(), [
            'title' => 'required',
            'state' => 'required',
            'external_id' => 'nullable',
            'supervisor' => 'required',
            'description' => 'nullable',
            'address' => 'required',
            'latitude' => 'required',
            'longitude' => 'required',
            'area_radius' => 'required',
            'project' => 'nullable'
        ]);

        

        if ($validator->fails()) {
            $error = $validator->errors()->first();
            return response()->json([
                'message' => $error
            ], 422);
        }else{

        $validatedData = $validator->validated();
        $site = Sites::findOrFail($id);
       
        $updateData = [
            'title' => $validatedData['title'],
            'site_state' => $validatedData['state'],
            'external_id' => $validatedData['external_id'],
            'supervisor_id' => $validatedData['supervisor'],
            'description' => $validatedData['description'],
            'street_address' => $validatedData['address'],
            'latitude' => $validatedData['latitude'],
            'longitude' => $validatedData['longitude'],
            'area_radius' => $validatedData['area_radius']
        ];

        $site->update($updateData);
       
        $getuserdetail = DB::table('emp_personal_details')->where('emp_id',auth()->user()->id)->first();
        $siteTitle = e($site->title ?? 'Global Site');
        $description = "<a href='".url('/')."/user-profile/".Auth::user()->id."' style='text-transform: capitalize;' role='button' class='primary text-decoration-none'> {$getuserdetail->first_name} {$getuserdetail->middle_name} {$getuserdetail->last_name}</a>  change the site detail: '{$siteTitle}'."; 
        
        $history = [
            'site_id' => $site->id,
            'description' => $description,
            'updated_by' => auth()->user()->id,
        ];

        SiteHistory::create($history);
        session()->flash('success', 'Site Updated Successfully');
        return response()->json(['message' => 'Site Updated Successfully', 'Data' => $updateData]);
    }
    }

    public function updateStatus($id)
    {     
        
        $site = Sites::find($id);
        if (!$site) {
            return response()->json([
                'message' => 'Site not found',
            ], 404);
        }
        // Toggle the active status
        $site->active = $site->active == 1 ? 0 : 1;
        $site->save();
        session()->flash('success', 'Status Updated Successfully');
        return response()->json([
            'message' => 'Status Updated Successfully',
            'data' => [
                'active' => $site->active,
            ],
        ], 200);
    }

    public function destroy($id)
    {
        Sites::where('id',$id)->update(['del' => '1']);
         $site = Sites::where('id',$id)->first();

        $getuserdetail = DB::table('emp_personal_details')->where('emp_id',auth()->user()->id)->first();
        $siteTitle = e($site->title ?? 'Global Site');
        $description = "<a href='".url('/')."/user-profile/".Auth::user()->id."' style='text-transform: capitalize;' role='button' class='primary text-decoration-none'> {$getuserdetail->first_name} {$getuserdetail->middle_name} {$getuserdetail->last_name}</a>  Deleted that site: '{$siteTitle}'."; 
        
        $history = [
            'site_id' => $site->id,
            'description' => $description,
            'updated_by' => auth()->user()->id,
        ];
        session()->flash('success', 'Site Deleted Successfully');
        SiteHistory::create($history);
        return response()->json([
            'message' => 'Site Deleted Successfully'
        ],200);
    }


    public function view($id)
    {
        $site = Sites::where('id', $id)->first();
        $sitehistories = SiteHistory:: where('site_id', $id)->get();
      
        $emppersonal = EmpPersonalDetails::where('id', $site->created_by)->first();
        $empsuper = EmpPersonalDetails::where('emp_id', $site->supervisor_id)->first();
        return view('Sites.view', compact('site','emppersonal','empsuper','sitehistories'));
    }
    public function site_list()
    {
        $sites = Sites::where('active', 1)
            ->get(['id', 'title'])
            ->map(function($site) {
                return [
                    'label' => $site->title ?? 'Global Site',
                    'value' => $site->id,
                ];
            });
    
        if ($sites->isEmpty()) {
            return response()->json([
                'message' => 'No data found'
            ], 404);
        }
    
        return response()->json($sites, 200);
    }
    

}
